using System;
using System.Data;
using System.Collections;

using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/22/2004</CreationDate>
	///	<Note>
	///		The Food and Drug Administration classifies this software as a medical device.  
	///		As such, it may not be changed in any way. Modifications to this software may result 
	///		in an adulterated medical device under 21CFR820, the use of which is considered to be 
	///		a violation of US Federal Statutes.  Acquiring and implementing this software through 
	///		the Freedom of information Act requires the implementor to assume total responsibility 
	///		for the software, and become a registered manufacturer of a medical device, 
	///		subject to FDA regulations.
	///	</Note>
	///<summary>Performs an exchange of patient data with VistA system via VistALink.</summary>

	#endregion

	public class PatientData
	{
		private const string PATIENT_LOOKUP_RPC_NAME = "VBECS PATIENT LOOKUP";
		private const string SEARCH_VALUE_PARAMETER_NAME = "SEARCH_VALUE";
		private const string SEARCH_TYPE_PARAMETER_NAME = "SEARCH_TYPE";
		
		private const string SEARCH_TYPE_NAME = "NAME";
		private const string SEARCH_TYPE_SSN4 = "SSN4";
		private const string SEARCH_TYPE_SSN = "SSN";
		private const string SEARCH_TYPE_ICN = "ICN";

		private const string PATIENT_TableName = "Patient";

		private PatientData() {}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4157"> 
		///		<ExpectedInput>'A' as patient name, empty strings for other parameters.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with patients data retrieved from VistA.</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="0" testid ="4158"> 
		///		<ExpectedInput>Empty strings for all parameters.</ExpectedInput>
		///		<ExpectedOutput>Non-null object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4203"> 
		///		<ExpectedInput>Null patient name.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4204"> 
		///		<ExpectedInput>Null patient number.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="4208"> 
		///		<ExpectedInput>Null patient ICN.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5948"> 
		///		<ExpectedInput>Known VistA patient (XXZ,SEXZLT) who has expired in VistA.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with non-null PatientDeathDate column.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7029"> 
		///		<ExpectedInput>Patient number of 5740 for SSN Search using last 4 of SSN</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with patients data retrieved from VistA.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7030"> 
		///		<ExpectedInput>Patient number of 5740 and Patient Name of "A" for SSN Search using first character
		///		of last name and last 4 of SSN.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with patients data retrieved from VistA.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7031"> 
		///		<ExpectedInput>Patient number of 582355740 for SSN search.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with patients data retrieved from VistA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Queries VistA system executing a patient lookup and returns a read-only <see cref="DataSet"/>
		/// containing search results. Search is performed by one of supplied parameters.
		/// Implements BR_3.20
		/// </summary>
		/// <param name="patientName">Patient name to search by.</param>
		/// <param name="patientNumber">Patient Number to search by.</param>
		/// <param name="patientICN">Patient Integration Control Number (ICN) to search by.</param>
		/// <returns>Read-only <see cref="DataTable"/> containing patient lookup results.</returns>
		public static DataTable GetVistaPatientData( string patientName, string patientNumber, string patientICN )
		{
			if( patientName == null )
				throw( new ArgumentNullException( "patientName" ) );
            //CR 3515
			if( patientNumber == null )
				throw( new ArgumentNullException( "patientNumber" ) );

			if( patientICN == null )
				throw( new ArgumentNullException( "patientICN" ) );

			DataSet _ds = new DataSet();

			string _searchType = string.Empty;
			string _searchValue = string.Empty;
            //CR 3515
			if( patientNumber.Trim().Length == 4 )
			{
				_searchType = SEARCH_TYPE_SSN;
				_searchValue = patientNumber;
				if( patientName.Trim().Length == 1 )
				{
					_searchType = SEARCH_TYPE_SSN4;
					_searchValue = patientName + patientNumber;
				}
			}

            else if (patientNumber.Trim().Length > 4)
			{
				_searchType = SEARCH_TYPE_SSN;
                _searchValue = patientNumber;
			}

			else if( patientName.Trim().Length > 1 )
			{
				_searchType = SEARCH_TYPE_NAME;
				_searchValue = patientName;
			}
			else
			{
				_searchType = SEARCH_TYPE_ICN;
				_searchValue = patientICN;
			}

			_ds = GetVistaPatientData(_searchValue, _searchType);
			DataTable _dtPatient = GetPatientDataTable();

			foreach( DataTable dt in _ds.Tables )
			{
				if( dt.TableName == PATIENT_TableName )
				{
					foreach( DataRow dr in dt.Rows )
					{
						DataRow _drPatient = _dtPatient.NewRow();
						_drPatient[TABLE.Patient.PatientFirstName] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientLastName] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientMiddleName] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientSsn] = System.DBNull.Value;
						_drPatient[TABLE.Patient.SsnPseudoIndicator] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientDeathDate] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientIcn] = System.DBNull.Value;
						_drPatient[TABLE.Patient.VistaPatientId] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientDob] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientDobCode] = System.DBNull.Value;
						_drPatient[TABLE.Patient.PatientSexCode] = System.DBNull.Value;
						_dtPatient.Rows.Add(_drPatient);

						if( dr.Table.Columns.Contains(TABLE.Patient.VistaPatientId) && !dr.IsNull(TABLE.Patient.VistaPatientId) )
						{
							_drPatient[TABLE.Patient.VistaPatientId] = dr[TABLE.Patient.VistaPatientId].ToString();
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientLastName) && !dr.IsNull(TABLE.Patient.PatientLastName) )
						{
							_drPatient[TABLE.Patient.PatientLastName] = dr[TABLE.Patient.PatientLastName].ToString();
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientFirstName) && !dr.IsNull(TABLE.Patient.PatientFirstName) )
						{
							_drPatient[TABLE.Patient.PatientFirstName] = dr[TABLE.Patient.PatientFirstName].ToString();
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientMiddleName) && !dr.IsNull(TABLE.Patient.PatientMiddleName) )
						{
							_drPatient[TABLE.Patient.PatientMiddleName] = dr[TABLE.Patient.PatientMiddleName].ToString();
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientDob) && !dr.IsNull(TABLE.Patient.PatientDob) )
						{
							_drPatient[TABLE.Patient.PatientDob] = Common.HL7DateFormat.ConvertHL7DateTime(dr[TABLE.Patient.PatientDob].ToString());
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientDobCode) && !dr.IsNull(TABLE.Patient.PatientDobCode) )
						{
							_drPatient[TABLE.Patient.PatientDobCode] = dr[TABLE.Patient.PatientDobCode].ToString();
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientSsn) && !dr.IsNull(TABLE.Patient.PatientSsn) )
						{
							if( dr[TABLE.Patient.PatientSsn].ToString().EndsWith("P") )
							{
								_drPatient[TABLE.Patient.SsnPseudoIndicator] = 1;
								_drPatient[TABLE.Patient.PatientSsn] = dr[TABLE.Patient.PatientSsn].ToString().Remove(dr[TABLE.Patient.PatientSsn].ToString().Length-1, 1);

							}
							else
							{
								_drPatient[TABLE.Patient.PatientSsn] = dr[TABLE.Patient.PatientSsn].ToString();
							}
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientDeathDate) && !dr.IsNull(TABLE.Patient.PatientDeathDate) && dr[TABLE.Patient.PatientDeathDate].ToString().Length>0)
						{
							_drPatient[TABLE.Patient.PatientDeathDate] = Common.HL7DateFormat.ConvertHL7DateTime( dr[TABLE.Patient.PatientDeathDate].ToString() );
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientSexCode) && !dr.IsNull(TABLE.Patient.PatientSexCode) )
						{
							_drPatient[TABLE.Patient.PatientSexCode] = dr[TABLE.Patient.PatientSexCode].ToString();
						}
						if( dr.Table.Columns.Contains(TABLE.Patient.PatientIcn) && !dr.IsNull(TABLE.Patient.PatientIcn) )
						{
							_drPatient[TABLE.Patient.PatientIcn] = dr[TABLE.Patient.PatientIcn].ToString();
						}

					}
				}
			}

			return _dtPatient;
		}

		/// <summary>
		/// Executes VistA patient lookup RPC with a given parameters 
		/// and gets results in a read-only <see cref="DataSet"/>.
		/// </summary>
		/// <param name="searchValue">Search value.</param>
		/// <param name="searchType">Search type.</param>
		/// <returns>Read-only <see cref="DataSet"/> containing patient lookup results.</returns>
		private static DataSet GetVistaPatientData( string searchValue, string searchType )
		{
			string _rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetPatientLookupRpcRequest( searchValue, searchType ) );

			return DataTransformUtility.LoadXmlStringIntoDataSet( _rpcResult );
		}

		/// <summary>
		/// Creates patient lookup RPC request from a given search type and value.
		/// </summary>
		/// <param name="searchValue">Search value.</param>
		/// <param name="searchType">Search type.</param>
		/// <returns>Patient lookup RPC request.</returns>
		private static RpcRequest GetPatientLookupRpcRequest( string searchValue, string searchType )
		{
			RpcRequest _patientLookupRequest = VistALink.CreateRpcRequest( PATIENT_LOOKUP_RPC_NAME );
			_patientLookupRequest.Parameters.Add( GetPatientLookupRpcParameter( searchValue, searchType ) );

			return _patientLookupRequest;
		}

		/// <summary>
		/// Creates patient lookup RPC request parameter from supplied search type and value.
		/// </summary>
		/// <param name="searchValue">Search value.</param>
		/// <param name="searchType">Search type.</param>
		/// <returns>Patient lookup RPC request parameter.</returns>
		private static RpcArrayParameter GetPatientLookupRpcParameter( string searchValue, string searchType )
		{
			Hashtable _bufHt = new Hashtable();
			_bufHt.Add( SEARCH_VALUE_PARAMETER_NAME, searchValue );
			_bufHt.Add( SEARCH_TYPE_PARAMETER_NAME, searchType );

			return new RpcArrayParameter( 1, _bufHt );
		}

		private static DataTable GetPatientDataTable()
		{
			DataTable _dtPatient = new DataTable();

			//Patient DataTable
			_dtPatient.Columns.Add(TABLE.Patient.PatientFirstName);
			_dtPatient.Columns.Add(TABLE.Patient.PatientLastName);
			_dtPatient.Columns.Add(TABLE.Patient.PatientMiddleName);
			_dtPatient.Columns.Add(TABLE.Patient.PatientSsn);
			_dtPatient.Columns.Add(TABLE.Patient.SsnPseudoIndicator, typeof(byte));
			_dtPatient.Columns.Add(TABLE.Patient.PatientDeathDate, typeof(DateTime));
			_dtPatient.Columns.Add(TABLE.Patient.PatientIcn);
			_dtPatient.Columns.Add(TABLE.Patient.VistaPatientId);
			_dtPatient.Columns.Add(TABLE.Patient.PatientDob, typeof(DateTime));
			_dtPatient.Columns.Add(TABLE.Patient.PatientDobCode);
			_dtPatient.Columns.Add(TABLE.Patient.PatientSexCode);

			return _dtPatient;
		}
	}
}
